/*****************************************************************************
| FILE:         inc_fi_handler.cpp
| PROJECT:      G3g
| SW-COMPONENT: AmControllerPlugin
|------------------------------------------------------------------------------
| DESCRIPTION:  FI Messages over INC
|
|------------------------------------------------------------------------------
| COPYRIGHT:    (c) 2013 Robert Bosch GmbH
| HISTORY:
| Date      | Modification               | Author
| 03.06.13  | Initial revision           | Ranjit Katuri
| --.--.--  | ----------------           | -------, -----
|
|*****************************************************************************/

#include <assert.h>
#include "inc_fi_handler.h"

#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#include "generic_msgs_if.h"

#include <iostream>
using namespace std;

DLT_IMPORT_CONTEXT(incfi_dltlog)

/**
* Constructor
*/
inc_fi_handler::inc_fi_handler(uint16_t u16AppID, uint16_t u16ServiceID, uint16_t u16major, uint16_t u16minor)
:m_ptrmain(inc_fi_main::poGetInstance()),
m_u16appid(u16AppID),
m_u16serviceid(u16ServiceID),
m_u16regid(0xFFFF),
m_u16majorversion(u16major),
m_u16minorversion(u16minor),
m_u8servicestate(CCA_C_U8_SVCSTATUS_NOT_AVAILABLE),
m_fidupreglist()
{
  DLT_LOG(incfi_dltlog,DLT_LOG_INFO,DLT_STRING("New handler for srvid="),DLT_UINT(u16ServiceID),DLT_STRING(",AppID="),DLT_UINT(u16AppID));
  if(m_ptrmain != NULL)
  {
    //Register handler with dispatcher
    m_ptrmain->bRegisterHandler(m_u16serviceid, this);
  }
  //Initializing a regular mutex
  pthread_mutex_init(&m_upregmutex, NULL);
}

/**
* Destructor
*/
inc_fi_handler::~inc_fi_handler()
{
  if(m_ptrmain != NULL)
  {
    //Unregister handler
    m_ptrmain->bUnRegisterHandler(m_u16serviceid);
    m_ptrmain = NULL;
  }
}

/**
* Register for a FID
*/
bool inc_fi_handler::bRegisterForFID(uint16_t u16FunctionID)
{
  pthread_mutex_lock(&m_upregmutex);
  m_fidupreglist.push_back(u16FunctionID);
  pthread_mutex_unlock(&m_upregmutex);

  //Now check and register if possible
  vCheckAndRegisterFID();

  return true;
}

/**
* Unregister for a FID
*/
bool inc_fi_handler::bUnRegisterForFID(uint16_t u16FunctionID)
{
  bool ret = false;

  if(bIfServiceAvailable())
  {
    //Create an empty message with opcode set to relupreg
    gm_tclEmptyMessage oRelUpRegMessage
      (
      APP_ID_INC_CCA_ADAPTER,        // AppID of this CCA INC Adapter
      m_u16appid,            // AppID of the other Server
      m_u16regid,                    // RegId for the service
      0,                            // always 0
      m_u16serviceid,            // the SID of the service
      u16FunctionID,                       // the FID to register for
      AMT_C_U8_CCAMSG_OPCODE_RELUPREG  // this is an RELUPREG-message!
      );

    if(m_ptrmain != NULL)
    {
      ret = m_ptrmain->bPostMessage(&oRelUpRegMessage);
    }
  }
  return ret;
}
/**
* Helper function to copy the data from a FI Message to amt_tclServiceData
*/
void inc_fi_handler::vCopyDatatoFIMessage(amt_tclServiceData* poFIMsg,fi_tclTypeBase &oFIData)
{
   fi_tclVisitorMessage oInVisitorMsg(poFIMsg);
   oInVisitorMsg.s32GetData(oFIData,m_u16majorversion);
}

/**
* Helper function post FI data
*/
bool inc_fi_handler::bPostFIMessage(uint16_t u16FID, uint8_t u8Opcode, fi_tclTypeBase& oOutData)
{
  bool ret = false;

  if(bIfServiceAvailable())
  {
    /*Check code here*//* check the 2nd parameter */
    fi_tclVisitorMessage oOutVisitorMsg(oOutData,m_u16majorversion);

    /* Set the CCA message information */
    oOutVisitorMsg.vInitServiceData
      (
      APP_ID_INC_CCA_ADAPTER,                   /* Source app-ID   */
      m_u16appid,                /* Dest. app-ID    */
      AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,        /* stream type     */
      0,                                        /* stream counter  */
      m_u16regid,                               /* Register ID     */
      0,                    /* Command counter, filling it as zero, it has to be filled up by INC CCA Adapter */
      m_u16serviceid,              /* Service-ID      */
      u16FID,                                   /* Function-ID     */
      u8Opcode,                                 /* Opcode          */
      0,                                        /* ACT             */
      0,                                        /* Source sub-ID   */
      0                                         /* Dest. sub-ID    */
      );

    if(m_ptrmain != NULL)
    {
      DLT_LOG(incfi_dltlog,DLT_LOG_INFO,DLT_STRING("Posting a FI Msg, FID="),DLT_UINT(u16FID),DLT_STRING(",Opcode="),DLT_UINT(u8Opcode));
      ret = m_ptrmain->bPostMessage(&oOutVisitorMsg);
    }
  }

   return ret;
}

/**
* Helper function to send a service register message
*/
bool  inc_fi_handler::bRegisterService()
{
  bool ret = false;

  uint16_t srcappid = APP_ID_INC_CCA_ADAPTER;

  amt_tclServiceRegister srvreg(srcappid, m_u16appid, m_u16serviceid, m_u16majorversion, m_u16minorversion);


  if(m_ptrmain != NULL)
  {
    DLT_LOG(incfi_dltlog,DLT_LOG_INFO,DLT_STRING("Posting a SRV Reg Msg, SRC="),DLT_UINT(srcappid),DLT_STRING(",Dest="),DLT_UINT(m_u16appid),\
      DLT_STRING(",ServiceID="),DLT_UINT(m_u16serviceid));
    ret = m_ptrmain->bPostMessage(&srvreg);
  }
  return ret;
}
/**
* Helper function to send a service register message
*/
bool  inc_fi_handler::bUnRegisterService()
{
  if(m_u16regid == 0xFFFF)
    return true;

  bool ret = false;

  uint16_t srcappid = APP_ID_INC_CCA_ADAPTER;

  amt_tclServiceUnregister srvunreg(srcappid, m_u16appid, m_u16serviceid, m_u16regid);

  if(m_ptrmain != NULL)
  {
    DLT_LOG(incfi_dltlog,DLT_LOG_INFO,DLT_STRING("Posting a SRV UNReg Msg, SRV="),DLT_UINT(m_u16serviceid));

    ret = m_ptrmain->bPostMessage(&srvunreg);
  }
  return ret;
}

/**
* Function invoked by main handler to set the service state
*/
void inc_fi_handler::vSetServiceState(uint16_t u16RegisterId, uint8_t u8servicestate, uint16_t /*u16SubId*/)
{
  if(m_ptrmain != NULL)

  DLT_LOG(incfi_dltlog,DLT_LOG_INFO,DLT_STRING("inc_fi_handler::vSetServiceState, SRV="),DLT_UINT(m_u16serviceid),DLT_STRING(",, RegId="),\
    DLT_UINT(u16RegisterId),DLT_STRING("Service state"),DLT_UINT(u8servicestate));

  //Update register ID
  m_u16regid =  u16RegisterId;

  if(m_u8servicestate != u8servicestate)
  {
    m_u8servicestate = u8servicestate;
    if(bIfServiceAvailable())
    {
      //Check and register any pending FIDs
      vCheckAndRegisterFID();
      vOnServiceAvailable();
    }
    else
    {
      vOnServiceUnavailable();
    }
  }
}

void inc_fi_handler::vCheckAndRegisterFID()
{
  if(bIfServiceAvailable())
  {
    pthread_mutex_lock(&m_upregmutex);
    for(vector<uint16_t>::iterator it = m_fidupreglist.begin(); it != m_fidupreglist.end(); it++)
    {
      //Create an empty message with opcode set to upreg
      gm_tclEmptyMessage oUpRegMessage
        (
        APP_ID_INC_CCA_ADAPTER,        // AppID of this CCA INC Adapter
        m_u16appid,            // AppID of the other Server
        m_u16regid,                    // RegId for the service
        0,                            // always 0
        m_u16serviceid,            // the SID of the service
        *it,                       // the FID to register for
        AMT_C_U8_CCAMSG_OPCODE_UPREG  // this is an Upreg-message!
        );

      if(m_ptrmain != NULL)
      {
        DLT_LOG(incfi_dltlog,DLT_LOG_INFO,DLT_STRING("UPREG, SRV="),DLT_UINT(m_u16serviceid),DLT_STRING(", FID="),DLT_UINT(*it));

        if(!m_ptrmain->bPostMessage(&oUpRegMessage))
        {
          DLT_LOG(incfi_dltlog,DLT_LOG_INFO,DLT_STRING("Failed to post msg"));
        }
      }
    }
    m_fidupreglist.clear();
    pthread_mutex_unlock(&m_upregmutex);
  }
}


